/* Reconstructed Commander Keen 1-3 Source Code
 * Copyright (C) 2021-2025 K1n9_Duk3
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifndef __BORLANDC__
#pragma inline	// ScrollTextWindow uses inline assembly
#endif
#include "KEENDEF.H"

/*
==============================================================================

                  PREVIEWS, ABOUT ID & TITLE SCREEN ROUTINES
	
==============================================================================
*/


/*
=====================
=
= Previews
=
=====================
*/

#if VERSION >= VER_100
void Previews(void)
{
#if (EPISODE == 1)

	DrawPicFile("PREVIEW2.CK1");
	FadeIn();
	WaitVBL(300);
	FadeOut();
	DrawPicFile("PREVIEW3.CK1");
	FadeIn();
#if VERSION > VER_110
	ReadLevel(90);	// reload menu backgrounds (DrawPicFile trashes the level data!)
#endif
	WaitVBL(300);
	ShowText(previewsPtr, 0, 22);
#if VERSION == VER_110
	ReadLevel(90);	// reload menu backgrounds (DrawPicFile trashes the level data!)
#endif

#else	// EPISODE != 1

	DrawTextScreen(previewsPtr, 0, 22);
	FadeIn();
	ShowText(previewsPtr, 0, 22);
	originxglobal = 84*TILEGLOBAL;

#endif	// if EPISODE == 1 ... else ...
}
#endif	// if VERSION >= VER_100


/*
=====================
=
= ShowAboutId
=
=====================
*/

void ShowAboutId(void)
{
	ControlStruct ctrl;

	originxglobal = 63*TILEGLOBAL;
	originyglobal = 28*PIXGLOBAL;
	RF_ForceRefresh();
	RF_Clear();
	DrawIdScreen();
	FadeIn();
	ClearKeys();

	do
	{
		RF_Clear();
		RF_Refresh();
		if (HandleHotkeys())
			DrawIdScreen();

		ctrl = ControlPlayer(1);
	} while (!ctrl.button1 && !ctrl.button2 && !NoBiosKey(1));
}


/*
=====================
=
= DrawTitlePics
=
=====================
*/

void DrawTitlePics(void)
{
	DrawPic(8, 1, PIC_TITLE);
#if (EPISODE == 3)
	DrawPic(19, 182, PIC_F1HELP);
#else
	DrawPic(16, 182, PIC_F1HELP);
#endif
}


/*
=====================
=
= DrawTitleScreen
=
=====================
*/

void DrawTitleScreen(void)
{
	refreshhook = &DrawTitlePics;
#if VERSION > VER_100
	RF_Clear();
#endif
	RF_Refresh();
	RF_Refresh();
	refreshhook = NULL;
	ClearKeys();
}


/*
=====================
=
= DrawAboutId
=
=====================
*/

void DrawAboutId(void)
{
#if VERSION <= VER_110

#if VERSION < VER_100
	DrawPic(18, 23, PIC_IDLOGO);
#else
	DrawPic(17, 23, PIC_IDLOGO);
#endif
	sx = leftedge = 7;
	sy = 9;
	DrawText("We are a group of Software Artists\n");
	DrawText("whose goal is to bring commercial\n");
	DrawText("quality software to the public\n");
	DrawText("at shareware prices.\n\n");
	DrawText("Our effort is only possible with\n");
	DrawText("your support. Without it, we cannot\n");
	DrawText("continue to make this fine\n");
	DrawText("software so affordable.\n\n");
	DrawText("Thank you in advance for your\n");
	DrawText("contribution to the future of the\n");
	DrawText("growing shareware market.");
	
#else
	
	char lines[11][40] =	// Note: making this 'static' would be more efficient!
	{
		"We are a group of Software Artists\n",
#if VERSION < VER_134
		"whose goal is to bring commercial\n",
		"quality software to the public\n",
		"at shareware prices.\n\n",
		"Our effort is only possible with\n",
		"your support. Without it, we cannot\n",
		"continue to make this fine\n",
		"software so affordable.\n\n",
		"Thank you in advance for your\n",
		"contribution to the future of the\n",
		"growing shareware market."
#else
		"whose goal is to bring high quality\n",
		"PC entertainment to the public at\n",
		"reasonable prices.\n\n",
		"Our effort is only possible with\n",
		"your support. Without it, we cannot\n",
		"continue to make this fine soft-\n",
		"ware so affordable.\n\n",
		"Thank you in advance for your pur-\n",
		"chase. We will continue to provide\n",
		"the PC market with great software."
#endif
	};
	Sint16 i;

	DrawPic(17, 23, PIC_IDLOGO);
	sx = leftedge = 7;
	sy = 9;
	for (i = 0; i < 11; i++)
	{
		DrawText(lines[i]);
	}
	
#endif
}


/*
=====================
=
= DrawIdScreen
=
=====================
*/

void DrawIdScreen(void)
{
	refreshhook = &DrawAboutId;
	RF_ForceRefresh();
	RF_Clear();
	RF_Refresh();
	RF_Refresh();
	refreshhook = NULL;
	ClearKeys();
}

/*
==============================================================================

                         HIGHSCORE SCREEN ROUTINES

==============================================================================
*/


/*
=====================
=
= DrawHighscores
=
=====================
*/

void DrawHighscores(void)
{
	Sint16 tx, ty, anim, i;
	char scorestr[12];

// Keen 3 doesn't show parts or cities saved, so names and scores are moved
// a bit further to the right to compensate for that:
#if (EPISODE == 3)
#define X 5
#else
#define X 0
#endif

	DrawPic(15,  7, PIC_HIGHSCOR);
	DrawPic( 9+X, 44, PIC_NAME);
	DrawPic(23+X, 44, PIC_SCORE);
#if (EPISODE == 1)
	DrawPic(33, 44, PIC_PARTS);
#elif (EPISODE == 2)
	DrawPic(33, 40, PIC_SAVED);
#endif

	for (i = 0; i < HIGHSCORE_NUMENTRIES; i++)
	{
		tx = originxglobal / TILEGLOBAL;	// why is this done inside the loop?
		ty = originyglobal / TILEGLOBAL;
		anim = i % 4;
		sy = i*2 + 8;

#if (EPISODE == 1)
		// icons for the collected ship parts are placed as animating tiles in the level:
		if (highscores.gotJoystick[i])
		{
			SETTILE(tx + 14, sy/2 + ty, 0, anim + 221);
		}
		if (highscores.gotBattery[i])
		{
			SETTILE(tx + 15, sy/2 + ty, 0, anim + 237);
		}
		if (highscores.gotVacuum[i])
		{
			SETTILE(tx + 16, sy/2 + ty, 0, anim + 241);
		}
		if (highscores.gotWhiskey[i])
		{
			SETTILE(tx + 17, sy/2 + ty, 0, anim + 245);
		}
#elif (EPISODE == 2)
		// draw number of cities saved:
		sx = 36;
		itoa(highscores.citiesSaved[i], scorestr, 10);
		DrawText(scorestr);
#endif

		// draw score and name on top of everything:
		ltoa(highscores.scores[i], scorestr, 10);
		sx = 29+X - strlen(scorestr);
		DrawText(scorestr);
		sx = 9+X;
		DrawText(highscores.names[i]);
	}

#undef X
}


/*
=====================
=
= DrawScoreScreen
=
=====================
*/

void DrawScoreScreen(void)
{
	originxglobal = 84*TILEGLOBAL;
	originyglobal = 2*TILEGLOBAL;
	RF_ForceRefresh();
	RF_Clear();
	refreshhook = &DrawHighscores;
	RF_Refresh();
	RF_Refresh();
	refreshhook = NULL;
}


/*
=====================
=
= ShowScoreScreen
=
=====================
*/

void ShowScoreScreen(void)
{
	ControlStruct ctrl;

	DrawScoreScreen();
	FadeIn();
	ClearKeys();

	do
	{
		RF_Clear();
		RF_Refresh();
		if (HandleHotkeys())
		{
			DrawScoreScreen();
		}

		ctrl = ControlPlayer(1);
	} while (!ctrl.button1 && !ctrl.button2 && !NoBiosKey(1));
}

/*
==============================================================================

                           SAVE & RESTORE ROUTINES

==============================================================================
*/


/*
=====================
=
= SaveMenu
=
=====================
*/

void SaveMenu(void)
{
	Sint16 save = 0;
	char c;
	char filename[13] = "SAVED?.";

	strcat(filename, _extension);
	if (!canSave)
	{
		ExpWin(22, 3);
		Print("You can SAVE the game\n");
		Print("ONLY on the World Map!\n");
		Print("    press a key:");
		ClearKeys();
		Ack();
		return;
	}

	do
	{
		ExpWin(20, 3);
		Print("Which game position\n");
		Print("do you want to save?\n");
		Print("    1-9 or ESC:");
		do
		{
			c = Get() & 0xFF;
		} while (c != CHAR_ESCAPE && !(c >= '1' && c <= '9'));

		if (c == CHAR_ESCAPE)
		{
			return;
		}

		filename[5] = c;
		if (access(filename, 0) == 0)
		{
			ExpWin(20, 3);
			Print("That game position\n");
			Print("already exists!\n");
			Print("Overwrite it?:");
			do
			{
				c = toupper(Get() & 0xFF);
			} while(c != CHAR_ESCAPE && c != 'Y' && c != 'N');
			
			if (c == CHAR_ESCAPE)
			{
				return;
			}

			if (c == 'Y')
			{
				save++;
			}
		}
		else
		{
			save++;
		}
	} while (save == 0);

	// prepare gamestate and save it:
	gamestate.worldoriginx = worldCamX;
	gamestate.worldoriginy = worldCamY;
	gamestate.worldx = worldkeenx;
	gamestate.worldy = worldkeeny;
	SaveFile(filename, MK_FP(FP_SEG(&gamestate), FP_OFF(&gamestate)), sizeof(gamestate));

#if VERSION >= VER_100
	ExpWin(29, 3);
	Print("You can continue this game\n");
	Print("from the Main Menu next time\n");
	Print("you play. Press a key:");
	Get();
#endif
}


/*
=====================
=
= RestoreMenu
=
=====================
*/

boolean RestoreMenu(void)
{
#if VERSION < VER_100
#define WINW 20
#else
#define WINW 25
#endif
	char c;
	Sint32 size;
	char filename[13] = "SAVED?.";

	DrawChar(sx, sy*8, ' ');	// erase cursor from Main Menu

	do
	{
		ExpWin(WINW, 2);
		oldx = sx;	// not sure why these are here...
		oldy = sy;
#if VERSION < VER_100
		Print("Continue Which Game?\n");
		Print("     1-9 or ESC:");
#else
		Print("  Continue Which Game?\n");
		Print("       1-9 or ESC:");
#endif
		do
		{
			c = Get() & 0xFF;
		} while (!(c >= '1' && c <= '9') && c != CHAR_ESCAPE);

		if (c == CHAR_ESCAPE)
		{
			return false;
		}
		else
		{
			filename[5] = c;
			strcat(filename, _extension);
			size = Verify(filename);

			if (!size)
			{
				ExpWin(WINW, 2);
#if VERSION < VER_100
				Print("That game hasn't\n");
				Print("been saved yet!:");
#else
				Print("  That game hasn't\n");
				Print("  been saved yet!:");
#endif
				Get();
			}
			else if (size == sizeof(gamestate))
			{
				LoadFile(filename, (void *)&gamestate);
				restoredGame = true;
				return true;
			}
			else
			{
				ExpWin(WINW, 2);
				// The following text didn't fit into the old window size used in
				// the beta version, that's why the window width had to be increased
				// for version 1.0 and some leading spaces had to be added in the
				// other strings above.
				Print("That file is incompatible\n");
				Print("with this verion of CK:");
				Get();
			}
		}

	} while (true);
}

/*
==============================================================================

                        HELP TEXT & STORY TEXT ROUTINES

==============================================================================
*/


/*
=====================
=
= ShowStoryText
=
=====================
*/

void ShowStoryText(void)
{
	FadeOut();
	originxglobal = 42*TILEGLOBAL;
	originyglobal = 2*TILEGLOBAL;
	RF_ForceRefresh();
	RF_Refresh();
	FadeIn();
#if VERSION < VER_100
	PlayAnimation(storyanim);
	ShowText(storytxtPtr, 1, 20);
#else
	ShowText(storytxtPtr, 0, 16);
#endif
	FadeOut();
}


/*
=====================
=
= ShowHelpText
=
=====================
*/

void ShowHelpText(void)
{
	Sint32 x, y;

	x = originxglobal;
	y = originyglobal;
	// normalize screen position for ShowText:
	TILE_ALIGN(originxglobal);// = originxglobal & ~(TILEGLOBAL-1);
	TILE_ALIGN(originyglobal);// = originyglobal & ~(TILEGLOBAL-1);
	ShowText(helptextPtr, 1, 20);
	originxglobal = x;
	originyglobal = y;
}


/*
=====================
=
= DrawTextEx
=
=====================
*/

void DrawTextEx(boolean isRed, Sint16 x, Sint16 y, char *text)
{
	Sint16 oldsx, oldsy;

	// save cursor position:
	oldsx = sx;
	oldsy = sy;

	// move cursor to desited location:
	sx = x;
	sy = y;

	// draw text in the desired style:
	if (isRed)
	{
		DrawText(text);
	}
	else
	{
		Print(text);
	}

	// restore old cursor position:
	sx = oldsx;
	sy = oldsy;
}


/*
=====================
=
= DrawTextWindow
=
=====================
*/

void DrawTextWindow(void)
{
	Sint16 x;

	DrawWindow(4, textWindowMinY, 43, textWindowMaxY);

	// add a sub-section at the bottom of that window:
	DrawChar(4, (textWindowMaxY+1)*8, 4);
	DrawChar(43, (textWindowMaxY+1)*8, 4);
	DrawChar(4, (textWindowMaxY+2)*8, 1);
	DrawChar(43, (textWindowMaxY+2)*8, 3);
	for (x = 5; x < 43; x++)
	{
		DrawChar(x, (textWindowMaxY+2)*8, 2);
	}
	DrawTextEx(true, 5, textWindowMaxY+1, "       ESC to Exit / \x0F \x13 to Read      ");

	// draw the visible portion of the text inside the window:
	DrawTextLines(textWindowX, textWindowMinY+1, textdataPtr, textLineOffsets, textVisibleLines);
}


/*
=====================
=
= DrawTextScreen
=
=====================
*/

#if (VERSION >= VER_120)
void DrawTextScreen(char huge *textPtr, Sint16 minY, Sint16 maxY)
{
	Sint16 lines, height;

	height = maxY - minY - 1;
	lines = GetLineOffsets(textPtr, line_offsets, 38, MAXTEXTLINES);
	textWindowX = 5;
	textWindowMinY = minY;
	textdataPtr = textPtr;
	textLineOffsets = line_offsets;
	textVisibleLines = height;
	textWindowMaxY = maxY;

	refreshhook = &DrawTextWindow;
	RF_Refresh();
	RF_Refresh();
	refreshhook = NULL;
}
#endif


/*
=====================
=
= ShowText
=
=====================
*/

void ShowText(char huge *textPtr, Sint16 minY, Sint16 maxY)
{
	ControlStruct ctrl;
	Sint16 lines, line, height;

	height = maxY - minY - 1;
	lines = GetLineOffsets(textPtr, line_offsets, 38, MAXTEXTLINES);
	textWindowX = 5;
	textWindowMinY = minY;
	textdataPtr = textPtr;
	textLineOffsets = line_offsets;
	textVisibleLines = height;
	textWindowMaxY = maxY;

	refreshhook = &DrawTextWindow;
	RF_Refresh();
	RF_Refresh();
	refreshhook = NULL;

	line = 0;
	WaitVBL(8);

	do
	{
		ctrl = ControlPlayer(1);

		if (keydown[KEY_UP] || ctrl.dir == north)
		{
			if (line > 0)
			{
				line--;
				ScrollTextWindow(minY+1, maxY-1, 1);
				DrawTextLines(5, minY+1, textPtr, line_offsets+line, 1);
#if VERSION > VER_120
				WaitVBL(2);	// avoid scrolling too fast
#endif
			}
		}
		else if (keydown[KEY_DOWN] || ctrl.dir == south)
		{
			if (lines - height >= line && 200-height >= line)
			{
				line++;
				ScrollTextWindow(minY+1, maxY-1, 0);
				DrawTextLines(5, maxY-1, textPtr, line_offsets+line+height-1, 1);
#if VERSION > VER_120
				WaitVBL(2);	// avoid scrolling too fast
#endif
			}
		}

		if (keydown[KEY_PGUP])
		{
			if (line - height + 1 > 0)
			{
				line -= height - 1;
			}
			else
			{
				line = 0;
			}

			DrawTextLines(5, minY+1, textPtr, line_offsets+line, height);

			// wait until key is no longer pressed:
			while (keydown[KEY_PGUP]);	
		}
		if (keydown[KEY_PGDN])
		{
			if (line + height*2 < lines)
			{
				line += height - 1;
			}
			else
			{
				line = lines - height + 1;
			}

			DrawTextLines(5, minY+1, textPtr, line_offsets+line, height);

			// wait until key is no longer pressed:
			while (keydown[KEY_PGDN]);
		}
	} while (!keydown[KEY_ESCAPE] && !ctrl.button1 && !ctrl.button2);

	// wait until the buttons are no longer pressed:
	do
	{
		ctrl = ControlPlayer(1);
	} while (ctrl.button1 || ctrl.button2);
	ClearKeys();
}

/*
==============================================================================

                           APOGEE INTRO ROUTINES

==============================================================================
*/

// the code is re-using a text window variable for the Apogee intro:
#define apogeeY textWindowX
#if VERSION < VER_100
#define APOGEE_END_Y 75
#define APOGEE_START_DELAY 20
#else
#define APOGEE_END_Y 55
#define APOGEE_START_DELAY 30
#endif


/*
=====================
=
= DrawApogeePic
=
=====================
*/

void DrawApogeePic(void)
{
	DrawPic(16, apogeeY, PIC_APOGEE);
}


/*
=====================
=
= DrawIntroPics
=
=====================
*/

void DrawIntroPics(void)
{
#if VERSION < VER_100
	// "an APOGEE presentation":
	DrawPic(22, 50, PIC_AN);
	DrawPic(18, 130, PIC_PRESENT);
	DrawPic(16, apogeeY,    PIC_APOGEE);
#else
	// "an APOGEE presentation":
	DrawPic(22, apogeeY-10, PIC_AN);
	DrawPic(16, apogeeY,    PIC_APOGEE);
	DrawPic(18, apogeeY+32, PIC_PRESENT);

	// "of an ID SOFTWARE production":
	DrawPic(21,  99, PIC_OFAN);
	DrawPic(19, 114, PIC_IDSOFT);
	DrawPic(19, 159, PIC_PRODUCT);
#endif
}


/*
=====================
=
= ShowApogeeIntro
=
=====================
*/

void ShowApogeeIntro(void)
{
	Sint16 state, y;
	ControlStruct ctrl;
	Sint16 towait;
	Sint16 n;

	originxglobal = 104*TILEGLOBAL;
	originyglobal = 2*TILEGLOBAL;
#if VERSION < VER_100
	towait= 200;
#else
	towait= 300;
#endif
	y = 200;
	state = 0;
	n = 0;

	RF_ForceRefresh();
	ClearKeys();
#if VERSION >= VER_100
	forcemenu = false;
#endif

	while (towait--)
	{
		RF_Clear();
		apogeeY = y;
		RF_Refresh();

		switch (state)
		{
		case 0:
			if (++n > APOGEE_START_DELAY)
			{
				n = 0;
				state++;
				refreshhook = &DrawApogeePic;
			}
			break;

		case 1:
			if (y > APOGEE_END_Y)
			{
				y--;
			}
			else
			{
				state++;
				refreshhook = &DrawIntroPics;
			}
		}

		ctrl = ControlPlayer(1);
		if (ctrl.button1 || ctrl.button2 || NoBiosKey(1))
		{
			towait = 0;
#if VERSION >= VER_100
			forcemenu = true;
#endif
		}
	}

	refreshhook = NULL;
}

/*
==============================================================================

                        ORDERING INFORMATION ROUTINES

==============================================================================
*/

#if VERSION < VER_134

/*
=====================
=
= ShowOrderingScreen
=
=====================
*/

void ShowOrderingScreen(boolean timed)
{
#if (EPISODE == 1) //---------------------------------------------------------

#if VERSION < VER_100
#define SWAPDELAY 1000
#else
#define SWAPDELAY 500
#endif

	register Sint16 frame, spr;
	ControlStruct ctrl;
	Sint16 sprites[2] = {SPR_YORPSTAND1, SPR_GARGSTAND1};
#if VERSION <= VER_120
	Sint32 screenys[2] = {159*PIXGLOBAL, 151*PIXGLOBAL};
#else
	Sint32 screenys[2] = {151*PIXGLOBAL, 143*PIXGLOBAL};
#endif
	Sint32 screenxs[2] = {144*PIXGLOBAL, 141*PIXGLOBAL};
	Sint16 n, towait, counter;

	frame = 0;
	n = 0;
	spr = 0;
	counter = 0;

	DrawOrderingScreen();
	FadeIn();
	ClearKeys();
	towait = 2400;

	do
	{
		ctrl = ControlPlayer(1);
		RF_Clear();

		// update sprite animation:
		if (counter % 6 == 0)
		{
			if ((frame = frame + Rnd(3) - 2) > 7 || frame < 0)
			//if (frame > 7 || frame < 0)
			{
				frame = Rnd(7*7) / 7;
			}
		}
		// change alien (Yorp/Garg) every 1000/500 refreshs:
		if (++n > SWAPDELAY)
		{
			spr ^= 1;
			n = 0;
		}
		RF_PlaceSprite(screenxs[spr] + originxglobal, screenys[spr] + originyglobal, sprites[spr] + frame);

		RF_Refresh();
		counter++;

		if (timed)
		{
			towait = towait - tics;
		}

		if (HandleHotkeys())
		{
			ClearKeys();
			DrawOrderingScreen();
		}

		if (ctrl.button1 || ctrl.button2 || NoBiosKey(1))
		{
			towait = 0;
		}

	} while (towait > 0);

#undef SWAPDELAY

#elif (EPISODE == 2) //-------------------------------------------------------

	Sint16 baseshape, towait;
	ControlStruct ctrl;
	Sint16 shapes[] = {SPR_SCRUBR1, SPR_SCRUBD1, SPR_SCRUBL1, SPR_SCRUBU1};
	Sint32 scruby = 4, scrubx = 0, xspeed = 2, yspeed = 0;
	Sint16 frame = 0;
	Sint16 var_22 = 0, var_24 = 0;	//never used!
	Sint16 counter = 0;

	DrawOrderingScreen();
	FadeIn();
	ClearKeys();
	towait = 2400;
	do
	{
		ctrl = ControlPlayer(1);
		RF_Clear();

		if (xspeed > 0)
		{
			baseshape = shapes[0];
			scrubx += PIXEL_TO_GLOBAL(xspeed);
			if (scrubx > 302*PIXGLOBAL)
			{
				xspeed = 0;
				yspeed = 2;
			}
		}
		if (xspeed < 0)
		{
			baseshape = shapes[2];
			scrubx += PIXEL_TO_GLOBAL(xspeed);
			if (scrubx < 2)
			{
				xspeed = 0;
				yspeed = -2;
			}
		}
		if (yspeed > 0)
		{
			baseshape = shapes[1];
			scruby += PIXEL_TO_GLOBAL(yspeed);
			if (scruby > 184*PIXGLOBAL)
			{
				xspeed = -2;
				yspeed = 0;
			}
		}
		if (yspeed < 0)
		{
			baseshape = shapes[3];
			scruby += PIXEL_TO_GLOBAL(yspeed);
			if (scruby < 6)
			{
				xspeed = 2;
				yspeed = 0;
			}
		}

		if ((counter % 4) == 0)
		{
			frame ^= 1;
		}

		RF_PlaceSprite(scrubx+originxglobal, scruby+originyglobal, baseshape+frame);

		RF_Refresh();
		counter++;

		if (timed)
		{
			towait = towait - tics;
		}

		if (HandleHotkeys())
		{
			ClearKeys();
			DrawOrderingScreen();
		}

		if (ctrl.button1 || ctrl.button2 || NoBiosKey(1))
		{
			towait = 0;
		}

	} while (towait > 0);

#elif (EPISODE == 3) //-------------------------------------------------------

#define KEENX (originxglobal + 8*PIXGLOBAL)
#define KEENY (originyglobal + 176*PIXGLOBAL)
#define FOOBX (foobx + originxglobal)
#define FOOBY (originyglobal + 184*PIXGLOBAL)

	ControlStruct ctrl;
	Sint32 foobx, xspeed;
	register Sint16 hidetics = 100;
	register Sint16 movetics = 0;
	Sint16 var_10 = 0, var_12 = 0, var_14 = 0;	// never used!
	Sint16 towait;
	Sint16 counter = 0;


	DrawOrderingScreen();
	FadeIn();
	ClearKeys();
	towait = 2400;
	do
	{
		ctrl = ControlPlayer(1);
		RF_Clear();

		if (movetics == 0 || movetics > 10)
		{
			RF_PlaceSprite(KEENX, KEENY, SPR_KEENWALKR1);
		}
		else if (xspeed < 0)
		{
			RF_PlaceSprite(KEENX, KEENY, SPR_KEENSHOOTR);
		}
		else
		{
			RF_PlaceSprite(KEENX, KEENY, SPR_KEENWALKR1);
		}

		if (hidetics != 0)
		{
			hidetics--;
			if (hidetics == 0)
			{
				foobx = 340*PIXGLOBAL;
				xspeed = -PIXGLOBAL;
				movetics = Rnd(175) + 100;
			}
		}

		if (movetics != 0)
		{
			foobx += xspeed;
			if (xspeed < 0)
			{
				RF_PlaceSprite(FOOBX, FOOBY, (counter & 2)/2 + SPR_FOOBL1);
			}
			else
			{
				RF_PlaceSprite(FOOBX, FOOBY, (counter & 2)/2 + SPR_FOOBR1);
			}

			movetics--;
			if (movetics == 0)
			{
				if (xspeed < 0)
				{
					PlaySound(SND_YORPSCREAM);
					RF_Clear();
					RF_PlaceSprite(KEENX, KEENY, SPR_KEENSHOOTR);
					RF_PlaceSprite(FOOBX, FOOBY, SPR_FOOBYELL1);
					RF_Refresh();
					RF_Refresh();

					RF_Clear();
					RF_PlaceSprite(KEENX, KEENY, SPR_KEENSHOOTR);
					RF_PlaceSprite(FOOBX, FOOBY, SPR_FOOBYELL2);
					RF_Refresh();
					RF_Refresh();

					xspeed = 6*PIXGLOBAL;
					movetics = (340*PIXGLOBAL - foobx)/(6*PIXGLOBAL);
				}
				else
				{
					hidetics = Rnd(50) + 10;
					xspeed = 0;
				}
			}
		}

		RF_Refresh();
		counter++;

		if (timed)
		{
			towait = towait - tics;
		}

		if (HandleHotkeys())
		{
			ClearKeys();
			DrawOrderingScreen();
		}

		if (ctrl.button1 || ctrl.button2 || NoBiosKey(1))
		{
			towait = 0;
		}

	} while (towait > 0);

#undef KEENX
#undef KEENY
#undef FOOBX
#undef FOOBY

#endif //---------------------------------------------------------------------
}


/*
=====================
=
= DrawOrderingInfo
=
=====================
*/

void DrawOrderingInfo(void)
{
#if (EPISODE == 1)

	leftedge = sx = 12;
	sy = 4;
	DrawText("Commander Keen: Invasion\n");
	DrawText("of the Vorticons consists\n");
	DrawText("   of three unique and\n");
#if VERSION <= VER_120
	DrawText("  challenging episodes:\n\n");
	DrawText("1. Marooned on Mars   $15\n");
	DrawText("2. The Earth Explodes $15\n");
	DrawText("3. Keen Must Die!     $15\n\n");
	leftedge = sx = 4;
	DrawText(" Order the trilogy for $30 and you get:\n");
	DrawText("  * The \"Secret Hints & Tricks\" sheet\n");
	DrawText("  * The special \"cheat mode\" password\n");
	DrawText("  * The latest version of each game\n");
	DrawText("  * SEVERAL FREE BONUS GAMES!\n\n");
	DrawText("                       Mail orders to:\n");
	DrawText("(U.S. funds only       Apogee Software\n");
	DrawText("checks or M/O's        P.O. Box 476389\n");
	DrawText("include $2 P&H)        Garland, TX 75047\n\n");
#else
	DrawText("  challenging episodes:\n");
	DrawText("1. Marooned on Mars   $15\n");
	DrawText("2. The Earth Explodes $15\n");
	DrawText("3. Keen Must Die!     $15\n");
	leftedge = sx = 5;
#if VERSION == VER_132
	DrawText(" Order the trilogy for $20 and you get\n");
#else
	DrawText(" Order the trilogy for $30 and you get\n");
#endif
	leftedge = --sx;
	DrawText("  * The \"Secret Hints & Tricks\" sheet\n");
	DrawText("  * The special \"cheat mode\" password\n");
	DrawText("  * The latest version of each game\n");
#if VERSION == VER_132
	DrawText("  MENTION THIS GRAVIS PRE-REGISTRATION!\n\n");
#else
	DrawText("  * SEVERAL FREE BONUS GAMES!\n\n");
#endif
	DrawText("                       Mail orders to:\n");
	DrawText("(U.S. funds only       Apogee Software\n");
	DrawText("checks or M/O's        P.O. Box 476389\n");
	DrawText("include $2 P&H)        Garland, TX 75047\n\n\n");
	DrawText("Specify 5.25/3.5 disk size when ordering\n");
#endif
	sx = 0;
	DrawText("       Or order toll free: 1-800-852-5659    \n");

#elif (EPISODE == 2)

	leftedge = sx = 6;
	sy = 3;
#if VERSION <= VER_120
	DrawText("  Commander Keen: Invasion of the   \n");
	DrawText("Vorticons consists of three unique  \n");
	DrawText("     and challenging episodes:\n\n");
#else
	DrawText("   Commander Keen: Invasion of the  \n");
	DrawText(" Vorticons consists of three unique \n");
	DrawText("      and challenging episodes:\n\n");
#endif
	DrawText(" Order the trilogy for $30 and get:\n");
	DrawText("* The \"Secret Hints & Tricks\" sheet\n");
	DrawText("* The special \"cheat mode\" password\n");
	DrawText("* The latest version of each game\n");
	DrawText("* SEVERAL FREE BONUS GAMES!\n\n");
	DrawText("          Mail orders to:\n");
	DrawText("          Apogee Software\n");
	DrawText("          P.O. Box 476389\n");
	DrawText("          Garland, TX 75047\n\n");
	DrawText("  U.S. funds only; checks or M/O's\n");
#if VERSION <= VER_120
	DrawText("   Include $2 postage & handling\n\n");
#else
	DrawText("   Include $2 postage & handling\n");
	DrawText("Specify 5.25/3.5 disk when ordering.\n");
#endif
	DrawText(" Or order toll free: 1-800-852-5659 ");

#elif (EPISODE == 3)

	leftedge = sx = 4;
	sy = 4;
	DrawText("    Commander Keen: Invasion of the\n");
	DrawText("   Vorticons consists of three unique\n");
	DrawText("        and challenging episodes:\n\n");
	DrawText("   Order the trilogy for $30 and get:\n");
	DrawText("  * The \"Secret Hints & Tricks\" sheet\n");
	DrawText("  * The special \"cheat mode\" password\n");
	DrawText("  * The latest version of each game\n");
	DrawText("  * SEVERAL FREE BONUS GAMES!\n\n");
	DrawText(" Mail orders to:     U.S funds only;\n");
	DrawText(" Apogee Software     checks or M/O's.\n");
	DrawText(" P.O. Box 476389     Include $2 postage\n");
	DrawText(" Garland, TX 75047   and handling.\n\n");
#if VERSION > VER_120
	DrawText("Specify 5.25/3.5 disk size when ordering\n");
#endif
	DrawText("   Or order toll free: 1-800-852-5659 ");

#endif
}


/*
=====================
=
= DrawOrderingScreen
=
=====================
*/

void DrawOrderingScreen(void)
{
	originxglobal = 22*TILEGLOBAL;
	originyglobal = 2*TILEGLOBAL;
#if VERSION > VER_110
	// need to update tile origin for RF_PlaceSprite:
	originxtile = GLOBAL_TO_TILE(originxglobal);
	originytile = GLOBAL_TO_TILE(originyglobal);
#endif

#if (EPISODE == 2)
	originyglobal -= 4*PIXGLOBAL;	// BUG: should be adjusted BEFORE calculating originytile
#endif

	RF_ForceRefresh();
	RF_Clear();
	refreshhook = &DrawOrderingInfo;
	RF_Refresh();
	RF_Refresh();
	refreshhook = NULL;
	ClearKeys();
}

#endif	// if VERSION < VER_134

/*
=====================
=
= DrawText
=
=====================
*/

void DrawText(char *text)
{
	char ch;

	while ((ch = *text++) != 0)
	{
		if (ch == '\n')
		{
			sy++;
			sx = leftedge;
		}
		else if (ch == '\r')
		{
			sx = leftedge;
		}
		else
		{
			DrawChar(sx++, sy*8, ch + 0x80);	// using the later half of the font (red text on grey background)
		}
	}
}

/*
==============================================================================

                           MAPKEEN ROUTINES

==============================================================================
*/

objtype mapkeen;


/*
=====================
=
= ClipMapKeen
=
=====================
*/

boolean ClipMapKeen(objtype *ob)
{
	Sint16 extra;
	Sint16 x, y, tileleft, tileright, tiletop, tilebottom;
	boolean result;

#if VERSION > VER_100
	if (godmode)
	{
		return false;
	}
#endif

	ob->xmove = ob->xmove + ob->xspeed * tics;
	ob->ymove = ob->ymove + ob->yspeed * tics;

	// Note: 'mapkeen' is only used in this routine, so any
	// changes made to it are basically discarded on return.
	// Only the changes made to 'ob' will have an effect.
	mapkeen = *ob;
	result = false;

	mapkeen.bottom += mapkeen.ymove;
	mapkeen.top += mapkeen.ymove;
	tileleft = mapkeen.left / TILEGLOBAL;
	tileright = mapkeen.right / TILEGLOBAL;

	if (mapkeen.ymove > 0)
	{
		// moving down
		if (mapkeen.bottom / TILEGLOBAL != (mapkeen.bottom - mapkeen.ymove) / TILEGLOBAL)
		{
			// movement crossed a tile boundary, so check for blocking tiles (or levels):
			tilebottom = mapkeen.bottom / TILEGLOBAL;
			for (x = tileleft; x <= tileright; x++)
			{
				if (tile_blockDown[GETTILE(x,tilebottom,0)] || GETTILE(x,tilebottom,1) & infoBlockMask)
				{
					ob->yspeed = 0;
					extra = (mapkeen.bottom + 1) % TILEGLOBAL;
					ob->ymove -= extra;
					mapkeen.top -= extra;
					mapkeen.bottom -= extra;
					result = true;
					break;
				}
			}
		}
	}
	else if (mapkeen.ymove < 0)
	{
		// moving up
		if (mapkeen.top / TILEGLOBAL != (mapkeen.top - mapkeen.ymove) / TILEGLOBAL)
		{
			// movement crossed a tile boundary, so check for blocking tiles (or levels):
			tiletop = mapkeen.top / TILEGLOBAL;
			for (x = tileleft; x <= tileright; x++)
			{
				if (tile_blockUp[GETTILE(x,tiletop,0)] || GETTILE(x,tiletop,1) & infoBlockMask)
				{
					ob->yspeed = 0;
					extra = TILEGLOBAL - mapkeen.top % TILEGLOBAL;
					ob->ymove += extra;
					mapkeen.top += extra;
					mapkeen.bottom += extra;
					result = true;
					break;
				}
			}
		}
	}

	mapkeen.left += mapkeen.xmove;
	mapkeen.right += mapkeen.xmove;
	tiletop = mapkeen.top / TILEGLOBAL;
	tilebottom = mapkeen.bottom / TILEGLOBAL;

	if (mapkeen.xmove > 0)
	{
		// moving right
		if (mapkeen.right / TILEGLOBAL != (mapkeen.right - mapkeen.xmove) / TILEGLOBAL)
		{
			// movement crossed a tile boundary, so check for blocking tiles (or levels):
			tileright = mapkeen.right / TILEGLOBAL;
			for (y = tiletop; y <= tilebottom; y++)
			{
				if (tile_blockRight[GETTILE(tileright,y,0)] || GETTILE(tileright,y,1) & infoBlockMask)
				{
					ob->xspeed = 0;
					extra = (mapkeen.right + 1) % TILEGLOBAL;
					ob->xmove -= extra;
					mapkeen.left -= extra;
					mapkeen.right -= extra;
					result = true;
					break;
				}
			}
		}
	}
	else if (mapkeen.xmove < 0)
	{
		// moving left
		if (mapkeen.left / TILEGLOBAL != (mapkeen.left - mapkeen.xmove) / TILEGLOBAL)
		{
			// movement crossed a tile boundary, so check for blocking tiles (or levels):
			tileleft = mapkeen.left / TILEGLOBAL;
			for (y = tiletop; y <= tilebottom; y++)
			{
				if (tile_blockLeft[GETTILE(tileleft,y,0)] || GETTILE(tileleft,y,1) & infoBlockMask)
				{
					ob->xspeed = 0;
					extra = TILEGLOBAL - mapkeen.left % TILEGLOBAL;
					ob->xmove += extra;
					mapkeen.left += extra;
					mapkeen.right += extra;
					result |= true;
					break;
				}
			}
		}
	}
	return result;
}


/*
=====================
=
= ControlMapKeen
=
=====================
*/

void ControlMapKeen(ControlStruct ctrl, objtype *ob)
{
	Sint16 x, y, tileleft, tiletop, tileright, tilebottom;
	Sint16 spotX, spotY, move, anim;
	boolean blocked;

#if (EPISODE == 3)
	// do nothing if Keen is riding Messie:
	if (messiestate != 0)
	{
		return;
	}
#endif

	UpdateObjHitbox(ob);

	// check for level entrances (or teleporters):
	if (ctrl.button1 || ctrl.button2)
	{
		tileleft = ob->left / TILEGLOBAL;
		tileright = ob->right / TILEGLOBAL;
		tiletop = ob->top / TILEGLOBAL;
		tilebottom = ob->bottom / TILEGLOBAL;
		for (x = tileleft; x <= tileright; x++)
		{
			for (y = tiletop; y <= tilebottom; y++)
			{
				if (GETTILE(x, y, 1))
				{
					spotX = x;
					spotY = y;
					nextLevel = GETTILE(x, y, 1);
					if (nextLevel == 255)	// skip Keen's spawn point
						nextLevel = 0;
				}
			}
		}
	}

	// move mapkeen:
	ob->xmove = ob->ymove = 0;
	switch (ctrl.dir)
	{
	case northwest:
		ob->ymove = -4*PIXGLOBAL;
		ob->xmove = -4*PIXGLOBAL;
		ob->shapenum = SPR_MAPKEENU1;
		break;

	case north:
		ob->ymove = -4*PIXGLOBAL;
		ob->shapenum = SPR_MAPKEENU1;
		break;

	case northeast:
		ob->ymove = -4*PIXGLOBAL;
		ob->xmove = 4*PIXGLOBAL;
		ob->shapenum = SPR_MAPKEENU1;
		break;

	case east:
		ob->xmove = 4*PIXGLOBAL;
		ob->shapenum = SPR_MAPKEENR1;
		break;

	case southeast:
		ob->ymove = 4*PIXGLOBAL;
		ob->xmove = 4*PIXGLOBAL;
		ob->shapenum = SPR_MAPKEEND1;
		break;

	case south:
		ob->ymove = 4*PIXGLOBAL;
		ob->shapenum = SPR_MAPKEEND1;
		break;

	case southwest:
		ob->ymove = 4*PIXGLOBAL;
		ob->xmove = -4*PIXGLOBAL;
		ob->shapenum = SPR_MAPKEEND1;
		break;

	case west:
		ob->xmove = -4*PIXGLOBAL;
		ob->shapenum = SPR_MAPKEENL1;
		break;
	}
	//BUG? MapKeen speed doesn't adapt to frame rate

	move = false;
	if (ob->xmove | ob->ymove)	// yes, that's a bitwise OR in the original code!
	{
		anim = (((Uint16)timecount >> 4) & 3);
		move++;
	}
	else
	{
		anim = 0;
	}

	infoBlockMask = 0x8000;
#if VERSION < VER_100
	if (keydown[KEY_TAB])
#else
	if (keydown[KEY_TAB] && keydown[KEY_LSHIFT])
#endif
	{
		infoBlockMask = 0;
	}

	blocked = ClipMapKeen(ob);
	ob->x += ob->xmove;
	ob->y += ob->ymove;

	if (move && !(((Uint16)timecount >> 3) & 3))
	{
		if (blocked)
		{
			PlaySound(SND_WLDBLOCK);
		}
		else
		{
			PlaySound(SND_WLDWALK);
		}
	}

	// scroll screen and place mapkeen sprite:
	if (ob->xmove > 0 && ob->x - originxglobal > 11*TILEGLOBAL)
	{
		originxglobal += ob->xmove;
		if (originxglobal > originxmax)
		{
			originxglobal = originxmax;
		}
	}
	else if (ob->xmove < 0 && ob->x - originxglobal < 9*TILEGLOBAL)
	{
		originxglobal += ob->xmove;
		if (originxglobal < originxmin)
		{
			originxglobal = originxmin;
		}
	}
	if (ob->ymove > 0 && ob->y - originyglobal > 7*TILEGLOBAL)
	{
		originyglobal += ob->ymove;
		if (originyglobal > originymax)
		{
			originyglobal = originymax;
		}
	}
	else if (ob->ymove < 0 && ob->y - originyglobal < 3*TILEGLOBAL)
	{
		originyglobal += ob->ymove;
		if (originyglobal < originymin)
		{
			originyglobal = originymin;
		}
	}
	originxtile = GLOBAL_TO_TILE(originxglobal);
	originytile = GLOBAL_TO_TILE(originyglobal);

	RF_PlaceSprite(ob->x, ob->y, ob->shapenum+anim);

#if (EPISODE == 3)
	if (messiestate == 0)	// messiestate is always 0 here (see beginning of this function)
	{
		if (messiecooldown == 0)
		{
			Sint16 w, h;

			// check if Keen is touching the Messie sprite (assumes all Messie sprites have the same size):
			w = spritetable[SPR_MESSIELD1*4].width;	// width is in bytes (8 pixels per byte)
			h = spritetable[SPR_MESSIELD1*4].height;
			// Note: The check is a bit weird, as it operates on the size of the
			// Messie sprite and not its hitbox and it does not take the size of
			// the Keen sprite (nor its hitbox) into account.
			if (ob->x >= messiex && messiex+PIXEL_TO_GLOBAL(w*8l) >= ob->x
				&& ob->y >= messiey && ob->y <= messiey+(h << G_P_SHIFT))
			{
				// Keen starts riding Messie
				PlaySound(SND_CRYSTAL);
				messiestate++;
				messiecooldown = 30;
			}
		}
		else
		{
			messiecooldown--;
		}
	}
#endif

	// check for teleporters and BWB:
	if (nextLevel)
	{
		if (CheckMapKeenTiles(nextLevel, ob, spotX, spotY))
		{
			nextLevel = 0;
		}
	}
}


/*
=====================
=
= CheckMapKeenTiles
=
=====================
*/

#pragma argsused
boolean CheckMapKeenTiles(Sint16 num, objtype *ob, Sint16 spotX, Sint16 spotY)
{
#if (EPISODE == 1)

	Sint16 i;
	ControlStruct ctrl;
	Sint16 t1, t2, anim, basetile, x, y;

	if (num == 20)
	{
		ExpWin(20, 8);
		Print("Your ship is missing\nthese parts:\n\n\n\n\n");
		Print("    GO GET THEM!\n");
		Print("   press a ");
		switch (playermode[1])
		{
		case keyboard:
			Print("key:");
			break;

		default:
			Print("button:");
			break;
		}
		if (!gamestate.gotJoystick)
		{
			DrawTile(leftedge+3, (sy-4)*8, 321);
		}
		if (!gamestate.gotBattery)
		{
			DrawTile(leftedge+7, (sy-4)*8, 322);
		}
		if (!gamestate.gotVacuum)
		{
			DrawTile(leftedge+11, (sy-4)*8, 323);
		}
		if (!gamestate.gotWhiskey)
		{
			DrawTile(leftedge+15, (sy-4)*8, 324);
		}

		WaitVBL(15);
		Ack();
		RF_ForceRefresh();
		return true;	// cannot enter a level here
	}

	for (i = 0; i < 3; i++)	// probably a bug
	{
		if ((num & 0x20) == 0x20)
		{
			// decode teleporter indices:
			t1 = (num & 3) - 1;
			t2 = ((num >> 2) & 3) - 1;
			// BUG? negative indices are possible but not accounted for!

			x = spotX;
			y = spotY;

			basetile = 338;
			if (warpspots[t2].tag)
			{
				basetile = 342;
			}

			PlaySound(SND_TELEPORT);

			// animate the teleporter tile:
			for (i = 0; i < 16; i++)	// same loop variable as outer loop!
			{
				RF_Clear();
				RF_ForceRefresh();	// not really required here...

				if (i % 2 == 0)
				{
					if (++anim > 3)
					{
						anim = 0;
					}
				}
				SETTILE(x,y,0,anim + basetile);

				RF_Refresh();
			}

			// put "inactive" teleporter tile back on the map:
			basetile = 325;
			if (warpspots[t2].tag)
			{
				basetile = 99;
			}
			SETTILE(x,y,0,basetile);

			// move to destination:
			ob->x = oldx = worldkeenx = warpspots[t1].x;
			ob->y = oldy = worldkeeny = warpspots[t1].y;
			originxglobal = ob->x - 9*TILEGLOBAL;
			originyglobal = ob->y - 3*TILEGLOBAL;
			// Note: originxtile and originytile should be updated as well,
			// otherwise RF_PlaceSprite won't work correctly for moving or
			// animated sprites!

			x = ob->x / TILEGLOBAL;
			y = ob->y / TILEGLOBAL;

			basetile = 338;
			if (warpspots[t1].tag)
			{
				basetile = 342;
			}

			// animate the teleporter tile:
			for (i = 0; i < 16; i++)	// same loop variable as outer loop!
			{
				RF_Clear();
				RF_ForceRefresh();	// not really required here...

				if (i % 2 == 0)
				{
					if (++anim > 3)
					{
						anim = 0;
					}
				}
				SETTILE(x,y,0,anim + basetile);

				RF_Refresh();
			}

			// put "inactive" teleporter tile back on the map:
			basetile = 325;
			if (warpspots[t1].tag)
			{
				basetile = 99;
			}
			SETTILE(x,y,0,basetile);

			// wait for the player to release the buttons,
			// so Keen won't enter the teleporter right away:
			do
			{
				RF_Clear();
				ctrl = ControlPlayer(1);
				RF_PlaceSprite(ob->x, ob->y, ob->shapenum);
				RF_Refresh();
				HandleUserKeys();
				HandleHotkeys();
			} while (ctrl.button1 || ctrl.button2);

			return true;	// cannot enter a level here
		}
	}

#elif (EPISODE == 3)

	if (num == 20)
	{
		Sint16 message, y;

		message = Rnd(3);
		ExpWin(32, 6);
		y = sy;
		switch (message)
		{
		case 0:
			Print("You enter your ship, sit around\n");
			Print("for a while, get bored, then\n");
			Print("remember that you have to find\n");
			Print("the Grand Intellect!");
			break;

		case 1:
			Print("Into the ship you journey, lie\n");
			Print("about a bit, then resume your\n");
			Print("quest for the Grand Intellect!");
			break;

		case 2:
			Print("You feel like entering the ship\n");
			Print("and taking a rest, but the\n");
			Print("mystery of the Grand Intellect's\n");
			Print("identity changes your mind.");
			break;

		case 3:
			Print("Entering the ship might be a\n");
			Print("fun thing to do, but right now,\n");
			Print("you need to find the Grand\n");
			Print("Intellect and vanquish him!");
			break;
		}
		sy = y + 4;
		Print("\n         press a ");
		switch (playermode[1])
		{
		case keyboard:
			Print("key:");
			break;
		default:
			Print("button:");
		}
		ClearKeys();
		Ack();
		ClearKeys();
		RF_ForceRefresh();
		return true;	// cannot enter a level here
	}

	if ((num & 0xFF00) >= 0x2000 && (num & 0xFF00) <= 0x2200)
	{
		// info value is Messie-related data
		return true;	// cannot enter a level here
	}

	if ((num & 0xF00) == 0xF00)
	{
		Sint16 i, index, anim, tilenum;

		// decode source teleporter index:
		index = (num & 0xF0) >> 4;

		// animate the teleporter tile (and Messie):
		for (i = 0; i < 2; i++)
		{
			PlaySound(SND_TELEPORT);
			for (tilenum = 130; tilenum < 134; tilenum++)
			{
				SETTILE(warps[index].x, warps[index].y, 0, tilenum);
				RF_Clear();
				RF_PlaceSprite(messiex, messiey, messieshape + (anim & 2)/2);
				RF_Refresh();
				anim++;
				WaitVBL(4);
			}
		}
		// put "inactive" teleporter tile back on the map:
		SETTILE(warps[index].x, warps[index].y, 0, 134);

		// decode destination teleporter index:
		index = num & 0xF;

		// move screen to destination:
		if (warps[index].x < 10)
		{
			originxglobal = 2*TILEGLOBAL;
		}
		else
		{
			originxglobal = TILE_TO_GLOBAL(warps[index].x-10);
		}
		
		if (warps[index].y < 6)
		{
			originyglobal = 2*TILEGLOBAL;
		}
		else
		{
			originyglobal = TILE_TO_GLOBAL(warps[index].y-6);
		}
		
		// Note: originxtile and originytile should be updated as well,
		// otherwise RF_PlaceSprite won't work correctly for moving or
		// animated sprites!

		// animate the teleporter tile (and Messie):
		for (i = 0; i < 2; i++)
		{
			PlaySound(SND_TELEPORT);
			for (tilenum = 130; tilenum < 134; tilenum++)
			{
				SETTILE(warps[index].x, warps[index].y, 0, tilenum);
				RF_Clear();
				RF_PlaceSprite(messiex, messiey, messieshape + (anim & 2)/2);
				RF_Refresh();
				anim++;
				WaitVBL(4);
			}
		}
		// put "inactive" teleporter tile back on the map:
		SETTILE(warps[index].x, warps[index].y, 0, 134);

		// move Keen to destination:
		ob->x = TILE_TO_GLOBAL(warps[index].x);
		ob->y = TILE_TO_GLOBAL(warps[index].y);
		return true;	// cannot enter a level here
	}

#endif

	return false;
}


/*
=====================
=
= Ack
=
= Waits for key or joystick/mouse button (and animates the cursor)
=
=====================
*/

void Ack(void)
{
	ControlStruct c;
	Sint16 done, anim, i;

	anim = 0;
	done = false;
	
	//
	// This routine was by far the most complicated one to get an accurate
	// reconstruction of. The calls to ControlPlayer generate an automatic
	// hidden variable on the stack, whose contents are then copied into
	// whatever struct variable the result gets assigned to. The big problem
	// was that the original code used the opposite order of what the compiler
	// kept giving me when I compiled my reconstructed code. I used a fairly
	// convoluted version with goto statements that did give me the correct
	// machine code for Keen 1 and Keen 2, but it didn't work for Keen 3,
	// because Keen 3 needs to use different compiler optimizations.
	//
	// In the end, simply adding an unused local struct variable in the inner
	// loop was enough to trick the compiler into generating the same code as
	// in the original executables. So if you ever encounter a similar problem
	// when reconstructing Turbo C++ 1.0 code, that's something you could try.
	//

	// animate cursor and wait for a key or button press:
	do
	{
		DrawChar(sx, sy*8, anim+9);
		for (i = 0; i < 6; i++)
		{
			ControlStruct temp;	// never used, just here to trick the compiler
			
			WaitVBL(1);
			c = ControlPlayer(1);
			if (c.button1 || c.button2 || NoBiosKey(1))
			{
				done++;
				break;
			}
		}
		if (++anim > 4)
		{
			anim = 0;
		}
	} while (!done);

	// wait until buttons are no longer held down:
	do
	{
		c = ControlPlayer(1);
	} while (c.button1 | c.button2);	// yes, this is a bitwise or in the code
	
	ClearKeys();
}


/*
==============================================================================

                        SCROLLING TEXT VIEW ROUTINES

==============================================================================
*/


/*
=====================
=
= DrawTextLines
=
=====================
*/

void DrawTextLines(Sint16 x, Sint16 y, char huge *textPtr, linetype *lineoffs, Uint16 numlines)
{
	Uint16 i, line, off, len, fontoff;

	sx = x;
	sy = y;
	fontoff = 0;	// black & white font by default
	for (line = 0; line < numlines; line++)
	{
		off = lineoffs[line].off;
		if (off == 0xFFFF)
		{
			return;
		}
		len = lineoffs[line].len;

		if (*(textPtr+off+len-1) == '\r')
		{
			len--;
		}
		if (textPtr[off] == '~')
		{
			fontoff = 0x80;	// red & grey font
			off++;
			len--;
		}
		else
		{
			fontoff = 0;	// black & white font
		}

		// draw the line of text:
		for (i = 0; i < len; i++)
		{
			DrawChar(sx++, sy*8, (*(textPtr+off+i) + fontoff) & 0xFF);
		}
		// blank the rest of the line (if any):
		if (sx < 43)
		{
			CharBar(sx, sy, 42, sy, fontoff + ' ');
		}

		sy++;
		sx = x;
	}
}


/*
=====================
=
= PrepareText
=
=====================
*/

void PrepareText(char huge *textPtr)
{
	Uint16 i, len;
	char c;

#if VERSION >= VER_100
	if (textPtr == NULL)
	{
		Quit("Missing a text file!");
	}
#endif

	// This code assumes the text uses DOS-style line breaks ("\r\n"), so when
	// we see a '\r' character, we know that it will always be followed by a '\n'
	// character. The code also assumes that the text has a CTRL-Z character
	// (a.k.a. ASCII SUB, a.k.a. DOS End Of File, a.k.a. 0x1A) at the end.

	// get the length of the text, in bytes:
	for (len = 0; textPtr[len] != 0x1A; len++);

	// Remove any line breaks from the text, except for blank lines. Line breaks
	// can be forced by using a 0x1F character (CTRL-_) in the text. These forced
	// line breaks should only be placed at the end of a line in the text file.
	for (i = 0; textPtr[i] != 0x1A; i++)
	{
		c = textPtr[i];
		if (c == 0x1F)
		{
			// convert into a forced line break:
			textPtr[i] = '\r';
		}
		else if (c == '\r' && *(textPtr+i+2) != '\r')
		{
			// replace the line break (2 chars!) with a single space character:
			textPtr[i] = ' ';
			movedata(FP_SEG(textPtr+i+2), FP_OFF(textPtr+i+2), FP_SEG(textPtr+i+1), FP_OFF(textPtr+i+1), len-i);
			len--;
		}
		else if (c == '\r' && *(textPtr+i+2) == '\r')
		{
			// skip to the next non-blank line:
			while (textPtr[i] == '\r')
			{
				i += 2;
			}
			i--;	// because the for-loop will increase it at the end
		}
	}
}


/*
=====================
=
= GetLineOffsets
=
=====================
*/

Uint16 GetLineOffsets(char huge *textPtr, linetype *lineoffs, Uint16 width, Uint16 maxlines)
{
	Uint16 line, off;
	Uint16 linestart, state;

	linestart = 0;
	state = 0;
	line = 0;
	off = 0;

	do
	{
		state = 0;
		lineoffs[line].off = linestart;

		for (off = linestart; off < linestart + width; off++)
		{
			if (textPtr[off] == 0x1A)
			{
				state = 2;
				lineoffs[line].len = off - linestart;
				lineoffs[line+1].off = 0xFFFF;
				lineoffs[line+1].len = 0xFFFF;
				break;
			}
			else if (textPtr[off] == '\r')
			{
				lineoffs[line].len = off - linestart + 1;
				line++;
				linestart = off+2;	// always skip the character after the '\r' (usually '\n' or space)
				state++;
				break;
			}
		}

		if (state == 0)
		{
			// line is now too long, go back to last space character:
			for (; off > linestart; off--)
			{
				if (textPtr[off] == ' ')
				{
					lineoffs[line].len = off - linestart;
					line++;
					linestart = off+1;	// skip the space character
					break;
				}
			}
			if (off == linestart)
			{
				// didn't find a space character - use max width:
				lineoffs[line].len = width;
				line++;
				off += width;
				linestart += width;
			}
		}

		if (line == maxlines)
		{
			lineoffs[line-1].off = 0xFFFF;
			lineoffs[line-1].len = 0xFFFF;
			state = 2;
		}
	} while (state < 2);

	return line;
}


/*
=====================
=
= ScrollTextWindow
=
=====================
*/

// Note: This routine scrolls the full width of the video buffer up or down 
// by one character (8 pixels), so it's important to make sure the text window
// always fills the entire width of the displayed screen area.
void ScrollTextWindow(Sint16 minY, Sint16 maxY, Sint16 dir)
{
	Uint16 blocksize;

	blocksize = ((maxY-minY)*8) * SCREENWIDTH;
	switch (dir)
	{
	case 0:	// move graphics down (scroll up)
		outportb(GC_INDEX, GC_MODE);
		outportb(GC_INDEX+1, 1);
		outportb(SC_INDEX, SC_MAPMASK);
		outportb(SC_INDEX+1, 15);
		minY *= SCREENWIDTH*8;
		
		asm	pushf;
		asm	push	si;
		asm	push	di;
		asm	push	ds;
		asm	mov	di, minY;
		asm	mov	si, di;
		asm	add	si, SCREENWIDTH*8;
		asm	mov	cx, blocksize;
		asm	mov	ax, screenseg;
		asm	mov	es, ax;
		asm	mov	ds, ax;
		asm	cld;
		asm	rep movsb;
		asm	pop	ds;
		asm	pop	di;
		asm	pop	si;
		asm	popf;
		break;

	case 1:	// move graphics up (scroll down)
		outportb(GC_INDEX, GC_MODE);
		outportb(GC_INDEX+1, 1);
		outportb(SC_INDEX, SC_MAPMASK);
		outportb(SC_INDEX+1, 15);
		maxY *= SCREENWIDTH*8;
		maxY += SCREENWIDTH*8 - 1;
		
		asm	pushf;
		asm	push	si;
		asm	push	di;
		asm	push	ds;
		asm	mov	di, maxY;
		asm	mov	si, di;
		asm	sub	si, SCREENWIDTH*8;
		asm	mov	cx, blocksize;
		asm	mov	ax, screenseg;
		asm	mov	es, ax;
		asm	mov	ds, ax;
		asm	std;
		asm	rep movsb;
		asm	pop	ds;
		asm	pop	di;
		asm	pop	si;
		asm	popf;
		break;
	}
}


/*
=====================
=
= PlayAnimation
=
=====================
*/

#if VERSION < VER_100
void PlayAnimation(Sint16 *data)
{
	//Sint16 temp;
	animtype anims[MAXANIMS];
	Sint16 i;//, tx, ty, dirindex, screenx, screeny, shapenum, speed;
	
	for (i=0; i<MAXANIMS; i++)
	{
		anims[i].used = false;
	}
	
	while (*data && !bioskey(1))
	{
		switch (*data++)
		{
		case 1:
			// find an unused animation entry:
			for (i=0; i<MAXANIMS; i++)
			{
				if (!anims[i].used)
				{
					break;
				}
			}
			// Note: this will cause memory corruption when no unused animation
			// entry could be found!
			anims[i].used = true;
			anims[i].x = *data++;
			anims[i].y = *data++;
			anims[i].speed = *data++;
			anims[i].animdelay = *data++;
			anims[i].shapenums[0] = *data++;
			anims[i].shapenums[1] = *data++;
			anims[i].shapenums[2] = *data++;
			anims[i].shapenums[3] = *data++;
			anims[i].dirptr = *data++;
			anims[i].ID = *data++;
			anims[i].dirindex = 0;
			anims[i].frame = 0;
			break;
			
		case 2:
			switch (*data++)
			{
			case 0:
				{
					Sint16 tx, ty, tile;
					
					tx = *data++ + originxglobal / TILEGLOBAL;
					ty = *data++ + originyglobal / TILEGLOBAL;
					tile = *data++;
					mapplane[0][ty*mapwwide+tx] = tile;
				}
				break;
				
			case 1:
				Print((char *)(*data++));
				break;
				
			case 2:
				// set position for Print() -- in character units (8x8 pixel blocks)
				sx = *data++;
				sy = *data++;
				break;
				
			case 3:
				{
					Sint16 count;
#if 0
					count = *data++;
#else
					_BX = (Sint16)(data++);
					count = *((Sint16*)_BX);
#endif
					while (count--)
					{
						RF_Refresh();
					}
				}
				break;
				
			case 4:
				PlaySound(*data++);
				break;
				
			case 5:
				{
					Sint16 ID, temp;

#if 0					
					ID = *data++;
#else
					_BX = (Sint16)(data++);
					_AX = *((Sint16*)_BX);
					ID = _AX;
#endif
					for (temp=0; temp<MAXANIMS; temp++)
					{
						if (anims[temp].ID == ID)
						{
							switch (*data++)
							{
							case 0:
								anims[temp].x = *data++;
								break;
								
							case 1:
								anims[temp].y = *data++;
								break;
								
							case 2:
								anims[temp].speed = *data++;
								break;
								
							case 3:
								anims[temp].animdelay = *data++;
								break;
								
							case 4:
								anims[temp].shapenums[0] = *data++;
								break;
								
							case 5:
								anims[temp].shapenums[1] = *data++;
								break;
								
							case 6:
								anims[temp].shapenums[2] = *data++;
								break;
								
							case 7:
								anims[temp].shapenums[3] = *data++;
								break;
								
							case 8:
								anims[temp].dirptr = *data++;
								break;
								
							case 9:
								anims[temp].ID = *data++;
								break;
							}
							break;	// exit the for-loop
						}
					}
				}
				break;
			}
			break;
			
		case 3:
			{
				Sint16 frames;
#if 0
			frames = *data++;
#else
			_BX = (Sint16)(data++);
			_AX = *((Sint16*)_BX);
			frames = _AX;
#endif
			while (frames-- && !bioskey(1))
			{
				Sint16 temp;
				
				RF_Clear();
				for (temp = 0; temp < MAXANIMS; temp++)
				{
					if (anims[temp].used)
					{
						Sint16 ty, dirindex, screenx, screeny, shapenum, speed;
						
						ty = anims[temp].dirptr;
						dirindex = anims[temp].dirindex;
						screenx = anims[temp].x;
						screeny = anims[temp].y;
						speed = anims[temp].speed;
						switch (((Sint8*)ty)[dirindex])
						{
						case east:
							screenx += speed;
							break;
							
						case west:
							screenx -= speed;
							break;
							
						case north:
							screeny -= speed;
							break;
							
						case south:
							screeny += speed;
							break;
							
						case northwest:
							screenx -= speed;
							screeny -= speed;
							break;
							
						case northeast:
							screenx += speed;
							screeny -= speed;
							break;
							
						case southwest:
							screenx -= speed;
							screeny += speed;
							break;
							
						case southeast:
							screenx += speed;
							screeny += speed;
							break;
							
						case nodir+1:
							anims[temp].dirindex = -1;
							break;
						}
						anims[temp].dirindex++;
						anims[temp].x = screenx;
						anims[temp].y = screeny;
						
						if (frames % anims[temp].animdelay == 0)
						{
							if (++anims[temp].frame > 3)
							{
								anims[temp].frame = 0;
							}
						}
						shapenum = anims[temp].shapenums[anims[temp].frame];
						RF_PlaceSprite(screenx*PIXGLOBAL + originxglobal, screeny*PIXGLOBAL + originyglobal, shapenum);
					}
				}
				RF_Refresh();
			}
			break;
			}
		}
	}
	
	ClearKeys();
}
#endif	// VERSION < VER_100